/* WIN32CORE.xs
 *
 * (c) 1995 Microsoft Corporation. All rights reserved.
 * 		Developed by hip communications inc., http://info.hip.com/info/
 * Portions (c) 1993 Intergraph Corporation. All rights reserved.
 *
 *    You may distribute under the terms of either the GNU General Public
 *    License or the Artistic License, as specified in the README file.
 */
#include <windows.h>

#if defined(__CYGWIN__) && PERL_REVISION <= 5 && PERL_VERSION <= 6
# undef WIN32
#endif

#define _TEMP_WORD WORD

#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

#undef  WORD
#define WORD _TEMP_WORD

#if MODULE
#  include "ppport.h"
#endif

#if defined(__CYGWIN__)
#  undef  DllExport
#  define DllExport
#endif

#define ONE_K_BUFSIZE		1024

#ifndef get_childenv
#  define get_childenv		win32_get_childenv
#  define free_childenv		win32_free_childenv
#  define get_childdir		win32_get_childdir
#  define free_childdir		win32_free_childdir
#endif

#ifndef PerlDir_mapA
#  define PerlDir_mapA(dir)	dir
#endif

#define isSLASH(c)		((c) == '/' || (c) == '\\')

#define SKIP_SLASHES(s) \
    STMT_START {				\
	while (*(s) && isSLASH(*(s)))		\
	    ++(s);				\
    } STMT_END

#define COPY_NONSLASHES(d,s) \
    STMT_START {				\
	while (*(s) && !isSLASH(*(s)))		\
	    *(d)++ = *(s)++;			\
    } STMT_END

 
/* Add the CXT macros for Perls that don't have them. */

#if !defined(START_MY_CXT) && defined(PERL_IMPLICIT_CONTEXT)

/* Taken from perl.h in newer Perls */

#define START_MY_CXT

#define dMY_CXT_SV \
	SV *my_cxt_sv = *hv_fetch(PL_modglobal, MY_CXT_KEY,		\
				  sizeof(MY_CXT_KEY)-1, TRUE)

#define dMY_CXT	\
	dMY_CXT_SV;							\
	my_cxt_t *my_cxtp = INT2PTR(my_cxt_t*, SvUV(my_cxt_sv))

#define MY_CXT_INIT \
	dMY_CXT_SV;							\
	/* newSV() allocates one more than needed */			\
	my_cxt_t *my_cxtp = (my_cxt_t*)SvPVX(newSV(sizeof(my_cxt_t)-1));\
	Zero(my_cxtp, 1, my_cxt_t);					\
	sv_setuv(my_cxt_sv, PTR2UV(my_cxtp))

#define MY_CXT		(*my_cxtp)

#endif /* CXT macros */

/* Only use CXT if building backport */

#ifdef BACKPORT

#define MY_CXT_KEY	"Win32CORE::_guts" XS_VERSION

typedef struct {
	char	Wgetlogin_buffer[128];
	BOOL	Wuse_showwindow;
	WORD	Wshowwindow;
} my_cxt_t;
 
START_MY_CXT

#define w32_getlogin_buffer	MY_CXT.Wgetlogin_buffer
#define w32_use_showwindow	MY_CXT.Wuse_showwindow
#define w32_showwindow		MY_CXT.Wshowwindow

static DWORD w32_platform = (DWORD)-1;

#endif /* BACKPORT */

/* Don't build anything when compiled as an extension on Perls that have the
 * core linked in statically. Also don't build anything on non-windows
 * platformls.
 */

#if !defined(MODULE) || defined(BACKPORT) && (\
  defined(WIN32) || defined(_WIN32) || defined(__CYGWIN__) || \
  defined(__CYGWIN32__) || defined(__WINDOWS__) || defined(_WINCE) \
)

/* Internal functions */

DllExport unsigned long
win32_os_id(void)
{
    static OSVERSIONINFO osver;

    if (osver.dwPlatformId != w32_platform) {
	memset(&osver, 0, sizeof(OSVERSIONINFO));
	osver.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
	GetVersionEx(&osver);
	w32_platform = osver.dwPlatformId;
    }
    return (unsigned long)w32_platform;
}

DllExport int
IsWinNT(void)
{
    return (win32_os_id() == VER_PLATFORM_WIN32_NT);
}

DllExport int
IsWin95(void)
{
    return (win32_os_id() == VER_PLATFORM_WIN32_WINDOWS);
}

DllExport void*
win32_get_childenv(void)
{
    return NULL;
}

DllExport void
win32_free_childenv(void* d)
{
}

DllExport char*
win32_get_childdir(void)
{
    dTHX;
    char* ptr;
    char szfilename[(MAX_PATH+1)*2];
    if (USING_WIDE()) {
        WCHAR wfilename[MAX_PATH+1];
        GetCurrentDirectoryW(MAX_PATH+1, wfilename);
        W2AHELPER(wfilename, szfilename, sizeof(szfilename));
    }
    else {
        GetCurrentDirectoryA(MAX_PATH+1, szfilename);
    }

    New(0, ptr, strlen(szfilename)+1, char);
    strcpy(ptr, szfilename);
    return ptr;
}

DllExport void
win32_free_childdir(char* d)
{
    dTHX;
    Safefree(d);
}

/* Find the longname of a given path.  path is destructively modified.
 * It should have space for at least MAX_PATH characters. */
DllExport char *
win32_longpath(char *path)
{
    WIN32_FIND_DATA fdata;
    HANDLE fhand;
    char tmpbuf[MAX_PATH+1];
    char *tmpstart = tmpbuf;
    char *start = path;
    char sep;
    if (!path)
	return Nullch;

    /* drive prefix */
    if (isALPHA(path[0]) && path[1] == ':') {
	start = path + 2;
	*tmpstart++ = path[0];
	*tmpstart++ = ':';
    }
    /* UNC prefix */
    else if (isSLASH(path[0]) && isSLASH(path[1])) {
	start = path + 2;
	*tmpstart++ = path[0];
	*tmpstart++ = path[1];
	SKIP_SLASHES(start);
	COPY_NONSLASHES(tmpstart,start);	/* copy machine name */
	if (*start) {
	    *tmpstart++ = *start++;
	    SKIP_SLASHES(start);
	    COPY_NONSLASHES(tmpstart,start);	/* copy share name */
	}
    }
    *tmpstart = '\0';
    while (*start) {
	/* copy initial slash, if any */
	if (isSLASH(*start)) {
	    *tmpstart++ = *start++;
	    *tmpstart = '\0';
	    SKIP_SLASHES(start);
	}

	/* FindFirstFile() expands "." and "..", so we need to pass
	 * those through unmolested */
	if (*start == '.'
	    && (!start[1] || isSLASH(start[1])
		|| (start[1] == '.' && (!start[2] || isSLASH(start[2])))))
	{
	    COPY_NONSLASHES(tmpstart,start);	/* copy "." or ".." */
	    *tmpstart = '\0';
	    continue;
	}

	/* if this is the end, bust outta here */
	if (!*start)
	    break;

	/* now we're at a non-slash; walk up to next slash */
	while (*start && !isSLASH(*start))
	    ++start;

	/* stop and find full name of component */
	sep = *start;
	*start = '\0';
	fhand = FindFirstFile(path,&fdata);
	*start = sep;
	if (fhand != INVALID_HANDLE_VALUE) {
	    STRLEN len = strlen(fdata.cFileName);
	    if ((STRLEN)(tmpbuf + sizeof(tmpbuf) - tmpstart) > len) {
		strcpy(tmpstart, fdata.cFileName);
		tmpstart += len;
		FindClose(fhand);
	    }
	    else {
		FindClose(fhand);
		errno = ERANGE;
		return Nullch;
	    }
	}
	else {
	    /* failed a step, just return without side effects */
	    /*PerlIO_printf(Perl_debug_log, "Failed to find %s\n", path);*/
	    errno = EINVAL;
	    return Nullch;
	}
    }
    strcpy(path,tmpbuf);
    return path;
}

/* End of internal functions. */

/* XS Interface */

XS(w32_SetLastError)
{
    dXSARGS;
    if (items != 1)
	Perl_croak(aTHX_ "usage: Win32::SetLastError($error)");
    SetLastError(SvIV(ST(0)));
    XSRETURN_EMPTY;
}

XS(w32_GetLastError)
{
    dXSARGS;
    EXTEND(SP,1);
    XSRETURN_IV(GetLastError());
}

XS(w32_LoginName)
{
    dXSARGS;
#if defined(MODULE) && defined(BACKPORT) && !defined(PERL_IMPLICIT_SYS)
    dMY_CXT;
#endif
    char *name = w32_getlogin_buffer;
    DWORD size = sizeof(w32_getlogin_buffer);
    EXTEND(SP,1);
    if (GetUserName(name,&size)) {
	/* size includes NULL */
	ST(0) = sv_2mortal(newSVpvn(name,size-1));
	XSRETURN(1);
    }
    XSRETURN_UNDEF;
}

XS(w32_NodeName)
{
    dXSARGS;
    char name[MAX_COMPUTERNAME_LENGTH+1];
    DWORD size = sizeof(name);
    EXTEND(SP,1);
    if (GetComputerName(name,&size)) {
	/* size does NOT include NULL :-( */
	ST(0) = sv_2mortal(newSVpvn(name,size));
	XSRETURN(1);
    }
    XSRETURN_UNDEF;
}

XS(w32_DomainName)
{
    dXSARGS;
    HINSTANCE hNetApi32 = LoadLibrary("netapi32.dll");
    DWORD (__stdcall *pfnNetApiBufferFree)(LPVOID Buffer);
    DWORD (__stdcall *pfnNetWkstaGetInfo)(LPWSTR servername, DWORD level,
					  void *bufptr);

    if (hNetApi32) {
	pfnNetApiBufferFree = (DWORD (__stdcall *)(void *))
	    GetProcAddress(hNetApi32, "NetApiBufferFree");
	pfnNetWkstaGetInfo = (DWORD (__stdcall *)(LPWSTR, DWORD, void *))
	    GetProcAddress(hNetApi32, "NetWkstaGetInfo");
    }
    EXTEND(SP,1);
    if (hNetApi32 && pfnNetWkstaGetInfo && pfnNetApiBufferFree) {
	/* this way is more reliable, in case user has a local account. */
	char dname[256];
	DWORD dnamelen = sizeof(dname);
	struct {
	    DWORD   wki100_platform_id;
	    LPWSTR  wki100_computername;
	    LPWSTR  wki100_langroup;
	    DWORD   wki100_ver_major;
	    DWORD   wki100_ver_minor;
	} *pwi;
	/* NERR_Success *is* 0*/
	if (0 == pfnNetWkstaGetInfo(NULL, 100, &pwi)) {
	    if (pwi->wki100_langroup && *(pwi->wki100_langroup)) {
		WideCharToMultiByte(CP_ACP, 0, pwi->wki100_langroup,
				    -1, (LPSTR)dname, dnamelen, NULL, NULL);
	    }
	    else {
		WideCharToMultiByte(CP_ACP, 0, pwi->wki100_computername,
				    -1, (LPSTR)dname, dnamelen, NULL, NULL);
	    }
	    pfnNetApiBufferFree(pwi);
	    FreeLibrary(hNetApi32);
	    XSRETURN_PV(dname);
	}
	FreeLibrary(hNetApi32);
    }
    else {
	/* Win95 doesn't have NetWksta*(), so do it the old way */
	char name[256];
	DWORD size = sizeof(name);
	if (hNetApi32)
	    FreeLibrary(hNetApi32);
	if (GetUserName(name,&size)) {
	    char sid[ONE_K_BUFSIZE];
	    DWORD sidlen = sizeof(sid);
	    char dname[256];
	    DWORD dnamelen = sizeof(dname);
	    SID_NAME_USE snu;
	    if (LookupAccountName(NULL, name, (PSID)&sid, &sidlen,
				  dname, &dnamelen, &snu)) {
		XSRETURN_PV(dname);		/* all that for this */
	    }
	}
    }
    XSRETURN_UNDEF;
}

XS(w32_FsType)
{
    dXSARGS;
    char fsname[256];
    DWORD flags, filecomplen;
    if (GetVolumeInformation(NULL, NULL, 0, NULL, &filecomplen,
			 &flags, fsname, sizeof(fsname))) {
	if (GIMME_V == G_ARRAY) {
	    XPUSHs(sv_2mortal(newSVpvn(fsname,strlen(fsname))));
	    XPUSHs(sv_2mortal(newSViv(flags)));
	    XPUSHs(sv_2mortal(newSViv(filecomplen)));
	    PUTBACK;
	    return;
	}
	EXTEND(SP,1);
	XSRETURN_PV(fsname);
    }
    XSRETURN_EMPTY;
}

XS(w32_GetOSVersion)
{
    dXSARGS;
    OSVERSIONINFOA osver;

    if (USING_WIDE()) {
	OSVERSIONINFOW osverw;
	char szCSDVersion[sizeof(osverw.szCSDVersion)];
	osverw.dwOSVersionInfoSize = sizeof(OSVERSIONINFOW);
	if (!GetVersionExW(&osverw)) {
	    XSRETURN_EMPTY;
	}
	W2AHELPER(osverw.szCSDVersion, szCSDVersion, sizeof(szCSDVersion));
	XPUSHs(newSVpvn(szCSDVersion, strlen(szCSDVersion)));
	osver.dwMajorVersion = osverw.dwMajorVersion;
	osver.dwMinorVersion = osverw.dwMinorVersion;
	osver.dwBuildNumber = osverw.dwBuildNumber;
	osver.dwPlatformId = osverw.dwPlatformId;
    }
    else {
	osver.dwOSVersionInfoSize = sizeof(OSVERSIONINFOA);
	if (!GetVersionExA(&osver)) {
	    XSRETURN_EMPTY;
	}
	XPUSHs(newSVpvn(osver.szCSDVersion, strlen(osver.szCSDVersion)));
    }
    XPUSHs(newSViv(osver.dwMajorVersion));
    XPUSHs(newSViv(osver.dwMinorVersion));
    XPUSHs(newSViv(osver.dwBuildNumber));
    XPUSHs(newSViv(osver.dwPlatformId));
    PUTBACK;
}

XS(w32_IsWinNT)
{
    dXSARGS;
    EXTEND(SP,1);
    XSRETURN_IV(IsWinNT());
}

XS(w32_IsWin95)
{
    dXSARGS;
    EXTEND(SP,1);
    XSRETURN_IV(IsWin95());
}

XS(w32_FormatMessage)
{
    dXSARGS;
    DWORD source = 0;
    char msgbuf[ONE_K_BUFSIZE];

    if (items != 1)
	Perl_croak(aTHX_ "usage: Win32::FormatMessage($errno)");

    if (USING_WIDE()) {
	WCHAR wmsgbuf[ONE_K_BUFSIZE];
	if (FormatMessageW(FORMAT_MESSAGE_FROM_SYSTEM,
			  &source, SvIV(ST(0)), 0,
			  wmsgbuf, ONE_K_BUFSIZE-1, NULL))
	{
	    W2AHELPER(wmsgbuf, msgbuf, sizeof(msgbuf));
	    XSRETURN_PV(msgbuf);
	}
    }
    else {
	if (FormatMessageA(FORMAT_MESSAGE_FROM_SYSTEM,
			  &source, SvIV(ST(0)), 0,
			  msgbuf, sizeof(msgbuf)-1, NULL))
	    XSRETURN_PV(msgbuf);
    }

    XSRETURN_UNDEF;
}

XS(w32_Spawn)
{
    dXSARGS;
    char *cmd, *args;
    void *env;
    char *dir;
    PROCESS_INFORMATION stProcInfo;
    STARTUPINFO stStartInfo;
    BOOL bSuccess = FALSE;

    if (items != 3)
	Perl_croak(aTHX_ "usage: Win32::Spawn($cmdName, $args, $PID)");

    cmd = SvPV_nolen(ST(0));
    args = SvPV_nolen(ST(1));

    env = PerlEnv_get_childenv();
    dir = PerlEnv_get_childdir();

    memset(&stStartInfo, 0, sizeof(stStartInfo));   /* Clear the block */
    stStartInfo.cb = sizeof(stStartInfo);	    /* Set the structure size */
    stStartInfo.dwFlags = STARTF_USESHOWWINDOW;	    /* Enable wShowWindow control */
    stStartInfo.wShowWindow = SW_SHOWMINNOACTIVE;   /* Start min (normal) */

    if (CreateProcess(
		cmd,			/* Image path */
		args,	 		/* Arguments for command line */
		NULL,			/* Default process security */
		NULL,			/* Default thread security */
		FALSE,			/* Must be TRUE to use std handles */
		NORMAL_PRIORITY_CLASS,	/* No special scheduling */
		env,			/* Inherit our environment block */
		dir,			/* Inherit our currrent directory */
		&stStartInfo,		/* -> Startup info */
		&stProcInfo))		/* <- Process info (if OK) */
    {
	int pid = (int)stProcInfo.dwProcessId;
	if (IsWin95() && pid < 0)
	    pid = -pid;
	sv_setiv(ST(2), pid);
	CloseHandle(stProcInfo.hThread);/* library source code does this. */
	bSuccess = TRUE;
    }
    PerlEnv_free_childenv(env);
    PerlEnv_free_childdir(dir);
    XSRETURN_IV(bSuccess);
}

XS(w32_GetTickCount)
{
    dXSARGS;
    DWORD msec = GetTickCount();
    EXTEND(SP,1);
    if ((IV)msec > 0)
	XSRETURN_IV(msec);
    XSRETURN_NV(msec);
}

XS(w32_SetCwd)
{
    dXSARGS;
    if (items != 1)
	Perl_croak(aTHX_ "usage: Win32::SetCurrentDirectory($cwd)");
    if (!PerlDir_chdir(SvPV_nolen(ST(0))))
	XSRETURN_YES;

    XSRETURN_NO;
}

XS(w32_GetCwd)
{
    dXSARGS;
    /* Make the host for current directory */
    char* ptr = PerlEnv_get_childdir();
    /*
     * If ptr != Nullch
     *   then it worked, set PV valid,
     *   else return 'undef'
     */
    if (ptr) {
	SV *sv = sv_newmortal();
	sv_setpv(sv, ptr);
	PerlEnv_free_childdir(ptr);

#ifndef INCOMPLETE_TAINTS
	SvTAINTED_on(sv);
#endif

	EXTEND(SP,1);
	SvPOK_on(sv);
	ST(0) = sv;
	XSRETURN(1);
    }
    XSRETURN_UNDEF;
}

XS(w32_CopyFile)
{
    dXSARGS;
    BOOL bResult;
    if (items != 3)
	Perl_croak(aTHX_ "usage: Win32::CopyFile($from, $to, $overwrite)");
    if (USING_WIDE()) {
	WCHAR wSourceFile[MAX_PATH+1];
	WCHAR wDestFile[MAX_PATH+1];
	A2WHELPER(SvPV_nolen(ST(0)), wSourceFile, sizeof(wSourceFile));
	wcscpy(wSourceFile, PerlDir_mapW(wSourceFile));
	A2WHELPER(SvPV_nolen(ST(1)), wDestFile, sizeof(wDestFile));
	bResult = CopyFileW(wSourceFile, (WCHAR*)PerlDir_mapW(wDestFile), !SvTRUE(ST(2)));
    }
    else {
	char szSourceFile[MAX_PATH+1];
	strcpy(szSourceFile, PerlDir_mapA(SvPV_nolen(ST(0))));
	bResult = CopyFileA(szSourceFile, PerlDir_mapA(SvPV_nolen(ST(1))), !SvTRUE(ST(2)));
    }

    if (bResult)
	XSRETURN_YES;
    XSRETURN_NO;
}

XS(w32_GetFullPathName)
{
    dXSARGS;
    SV *filename;
    SV *fullpath;
    char *filepart;
    DWORD len;

    if (items != 1)
	Perl_croak(aTHX_ "usage: Win32::GetFullPathName($filename)");

    filename = ST(0);
    fullpath = sv_mortalcopy(filename);
    SvUPGRADE(fullpath, SVt_PV);
    if (!SvPVX(fullpath) || !SvLEN(fullpath))
        XSRETURN_UNDEF;

    do {
	len = GetFullPathName(SvPVX(filename),
			      SvLEN(fullpath),
			      SvPVX(fullpath),
			      &filepart);
    } while (len >= SvLEN(fullpath) && sv_grow(fullpath,len+1));
    if (len) {
	if (GIMME_V == G_ARRAY) {
	    EXTEND(SP,1);
	    if (filepart) {
		XST_mPV(1,filepart);
		len = filepart - SvPVX(fullpath);
	    }
	    else {
		XST_mPVN(1,"",0);
	    }
	    items = 2;
	}
	SvCUR_set(fullpath,len);
	ST(0) = fullpath;
	XSRETURN(items);
    }
    XSRETURN_EMPTY;
}

XS(w32_GetShortPathName)
{
    dXSARGS;
    SV *shortpath;
    DWORD len;

    if (items != 1)
	Perl_croak(aTHX_ "usage: Win32::GetShortPathName($longPathName)");

    shortpath = sv_mortalcopy(ST(0));
    SvUPGRADE(shortpath, SVt_PV);
    if (!SvPVX(shortpath) || !SvLEN(shortpath))
        XSRETURN_UNDEF;

    /* src == target is allowed */
    do {
	len = GetShortPathName(SvPVX(shortpath),
			       SvPVX(shortpath),
			       SvLEN(shortpath));
    } while (len >= SvLEN(shortpath) && sv_grow(shortpath,len+1));
    if (len) {
	SvCUR_set(shortpath,len);
	ST(0) = shortpath;
	XSRETURN(1);
    }
    XSRETURN_UNDEF;
}

XS(w32_GetLongPathName)
{
    dXSARGS;
    SV *path;
    char tmpbuf[MAX_PATH+1];
    char *pathstr;
    STRLEN len;

    if (items != 1)
	Perl_croak(aTHX_ "usage: Win32::GetLongPathName($pathname)");

    path = ST(0);
    pathstr = SvPV(path,len);
    strcpy(tmpbuf, pathstr);
    pathstr = win32_longpath(tmpbuf);
    if (pathstr) {
	ST(0) = sv_2mortal(newSVpvn(pathstr, strlen(pathstr)));
	XSRETURN(1);
    }
    XSRETURN_EMPTY;
}

XS(w32_GetNextAvailDrive)
{
    dXSARGS;
    char ix = 'C';
    char root[] = "_:\\";

    EXTEND(SP,1);
    while (ix <= 'Z') {
	root[0] = ix++;
	if (GetDriveType(root) == 1) {
	    root[2] = '\0';
	    XSRETURN_PV(root);
	}
    }
    XSRETURN_UNDEF;
}

XS(w32_SetChildShowWindow)
{
    dXSARGS;
#if defined(MODULE) && defined(BACKPORT) && !defined(PERL_IMPLICIT_SYS)
    dMY_CXT;
#endif
    BOOL use_showwindow = w32_use_showwindow;
    /* use "unsigned short" because Perl has redefined "WORD" */
    unsigned short showwindow = w32_showwindow;

    if (items > 1)
	Perl_croak(aTHX_ "usage: Win32::SetChildShowWindow($showwindow)");

    if (items == 0 || !SvOK(ST(0)))
        w32_use_showwindow = FALSE;
    else {
        w32_use_showwindow = TRUE;
        w32_showwindow = (unsigned short)SvIV(ST(0));
    }

    EXTEND(SP, 1);
    if (use_showwindow)
        ST(0) = sv_2mortal(newSViv(showwindow));
    else
        ST(0) = &PL_sv_undef;
    XSRETURN(1);
}

XS(w32_Sleep)
{
    dXSARGS;
    if (items != 1)
	Perl_croak(aTHX_ "usage: Win32::Sleep($milliseconds)");
    Sleep(SvIV(ST(0)));
    XSRETURN_YES;
}

void init_Win32CORE(void)
{
    dTHX;
    char *file = __FILE__;

    newXS("Win32::GetLastError", w32_GetLastError, file);
    newXS("Win32::SetLastError", w32_SetLastError, file);
    newXS("Win32::LoginName", w32_LoginName, file);
    newXS("Win32::NodeName", w32_NodeName, file);
    newXS("Win32::DomainName", w32_DomainName, file);
    newXS("Win32::FsType", w32_FsType, file);
    newXS("Win32::GetCwd", w32_GetCwd, file);
    newXS("Win32::SetCwd", w32_SetCwd, file);
    newXS("Win32::GetOSVersion", w32_GetOSVersion, file);
    newXS("Win32::FormatMessage", w32_FormatMessage, file);
    newXS("Win32::Spawn", w32_Spawn, file);
    newXS("Win32::GetTickCount", w32_GetTickCount, file);
    newXS("Win32::IsWinNT", w32_IsWinNT, file);
    newXS("Win32::IsWin95", w32_IsWin95, file);
    newXS("Win32::CopyFile", w32_CopyFile, file);
    newXS("Win32::GetFullPathName", w32_GetFullPathName, file);
    newXS("Win32::GetLongPathName", w32_GetLongPathName, file);
    newXS("Win32::GetNextAvailDrive", w32_GetNextAvailDrive, file);
    newXS("Win32::GetShortPathName", w32_GetShortPathName, file);
    newXS("Win32::SetChildShowWindow", w32_SetChildShowWindow, file);
    newXS("Win32::Sleep", w32_Sleep, file);
}

XS(boot_Win32CORE)
{
    dXSARGS;
#if defined(MODULE) && defined(BACKPORT) && !defined(PERL_IMPLICIT_SYS)
    MY_CXT_INIT;
#endif
    init_Win32CORE();
    XSRETURN_YES;
}

#else /* Do nothing in dynamic CORE extension and on non-Windows platforms */

XS(boot_Win32CORE)
{
}

#endif
